// Contact page specific JavaScript functionality

document.addEventListener('DOMContentLoaded', function() {
    initContactForm();
    initFAQ();
    initFormValidation();
});

// Contact form initialization
function initContactForm() {
    const form = document.getElementById('contact-form');
    
    if (form) {
        form.addEventListener('submit', handleFormSubmit);
        
        // Add real-time validation
        const inputs = form.querySelectorAll('input, select, textarea');
        inputs.forEach(input => {
            input.addEventListener('blur', validateField);
            input.addEventListener('input', clearFieldError);
        });
    }
}

// Handle form submission
async function handleFormSubmit(e) {
    e.preventDefault();
    
    const form = e.target;
    const submitBtn = form.querySelector('.submit-btn');
    const btnText = submitBtn.querySelector('.btn-text');
    const btnLoading = submitBtn.querySelector('.btn-loading');
    
    // Validate all fields
    if (!validateForm(form)) {
        return;
    }
    
    // Show loading state
    submitBtn.disabled = true;
    btnText.style.display = 'none';
    btnLoading.style.display = 'flex';
    
    // Collect form data
    const formData = new FormData(form);
    const formObject = {};
    formData.forEach((value, key) => {
        formObject[key] = value;
    });
    
    try {
        // Simulate form submission (replace with actual endpoint)
        await simulateFormSubmission(formObject);
        
        // Show success message
        showFormMessage('Thank you for your message! I\'ll get back to you within 24 hours.', 'success');
        
        // Reset form
        form.reset();
        
        // Track form submission (analytics)
        trackFormSubmission(formObject);
        
    } catch (error) {
        console.error('Form submission error:', error);
        showFormMessage('Sorry, there was an error sending your message. Please try again or contact me directly.', 'error');
    } finally {
        // Reset button state
        submitBtn.disabled = false;
        btnText.style.display = 'inline';
        btnLoading.style.display = 'none';
    }
}

// Simulate form submission (replace with actual API call)
function simulateFormSubmission(data) {
    return new Promise((resolve, reject) => {
        setTimeout(() => {
            // Simulate success/failure
            if (Math.random() > 0.1) { // 90% success rate
                resolve(data);
            } else {
                reject(new Error('Network error'));
            }
        }, 2000);
    });
}

// Form validation
function validateForm(form) {
    const fields = [
        { id: 'name', required: true, minLength: 2 },
        { id: 'email', required: true, type: 'email' },
        { id: 'phone', required: false, type: 'phone' },
        { id: 'subject', required: true },
        { id: 'message', required: true, minLength: 10 }
    ];
    
    let isValid = true;
    
    fields.forEach(field => {
        const input = form.querySelector(`#${field.id}`);
        if (!validateField({ target: input }, field)) {
            isValid = false;
        }
    });
    
    return isValid;
}

// Validate individual field
function validateField(e, fieldConfig = null) {
    const input = e.target;
    const value = input.value.trim();
    const fieldId = input.id;
    const errorElement = document.getElementById(`${fieldId}-error`);
    
    // Get field configuration
    const config = fieldConfig || getFieldConfig(fieldId);
    if (!config) return true;
    
    let errorMessage = '';
    
    // Required field validation
    if (config.required && !value) {
        errorMessage = `${getFieldLabel(fieldId)} is required`;
    }
    // Minimum length validation
    else if (config.minLength && value.length < config.minLength) {
        errorMessage = `${getFieldLabel(fieldId)} must be at least ${config.minLength} characters`;
    }
    // Email validation
    else if (config.type === 'email' && value && !isValidEmail(value)) {
        errorMessage = 'Please enter a valid email address';
    }
    // Phone validation
    else if (config.type === 'phone' && value && !isValidPhone(value)) {
        errorMessage = 'Please enter a valid phone number';
    }
    
    // Show/hide error
    if (errorMessage) {
        showFieldError(input, errorElement, errorMessage);
        return false;
    } else {
        clearFieldError(input, errorElement);
        return true;
    }
}

// Get field configuration
function getFieldConfig(fieldId) {
    const configs = {
        'name': { required: true, minLength: 2 },
        'email': { required: true, type: 'email' },
        'phone': { required: false, type: 'phone' },
        'subject': { required: true },
        'message': { required: true, minLength: 10 }
    };
    
    return configs[fieldId];
}

// Get field label
function getFieldLabel(fieldId) {
    const labels = {
        'name': 'Name',
        'email': 'Email',
        'phone': 'Phone',
        'subject': 'Subject',
        'message': 'Message'
    };
    
    return labels[fieldId] || fieldId;
}

// Show field error
function showFieldError(input, errorElement, message) {
    input.classList.add('error');
    if (errorElement) {
        errorElement.textContent = message;
        errorElement.style.display = 'block';
    }
}

// Clear field error
function clearFieldError(e) {
    const input = e.target;
    const errorElement = document.getElementById(`${input.id}-error`);
    
    input.classList.remove('error');
    if (errorElement) {
        errorElement.textContent = '';
        errorElement.style.display = 'none';
    }
}

// Email validation
function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

// Phone validation
function isValidPhone(phone) {
    const phoneRegex = /^[\+]?[1-9][\d]{0,15}$/;
    return phoneRegex.test(phone.replace(/[\s\-\(\)]/g, ''));
}

// Show form message
function showFormMessage(message, type) {
    // Remove existing message
    const existingMessage = document.querySelector('.form-message');
    if (existingMessage) {
        existingMessage.remove();
    }
    
    // Create new message
    const messageDiv = document.createElement('div');
    messageDiv.className = `form-message ${type}`;
    messageDiv.innerHTML = `
        <div class="message-content">
            <div class="message-icon">
                ${type === 'success' ? 
                    '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9 12L11 14L15 10M21 12C21 16.9706 16.9706 21 12 21C7.02944 21 3 16.9706 3 12C3 7.02944 7.02944 3 12 3C16.9706 3 21 7.02944 21 12Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>' :
                    '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M12 9V13M12 17H12.01M21 12C21 16.9706 16.9706 21 12 21C7.02944 21 3 16.9706 3 12C3 7.02944 7.02944 3 12 3C16.9706 3 21 7.02944 21 12Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>'
                }
            </div>
            <div class="message-text">${message}</div>
        </div>
    `;
    
    // Insert message
    const form = document.getElementById('contact-form');
    if (form) {
        form.insertBefore(messageDiv, form.firstChild);
        
        // Scroll to message
        messageDiv.scrollIntoView({ behavior: 'smooth', block: 'center' });
        
        // Auto-remove success messages after 5 seconds
        if (type === 'success') {
            setTimeout(() => {
                if (messageDiv.parentNode) {
                    messageDiv.remove();
                }
            }, 5000);
        }
    }
}

// FAQ functionality
function initFAQ() {
    const faqItems = document.querySelectorAll('.faq-item');
    
    faqItems.forEach(item => {
        const question = item.querySelector('.faq-question');
        const answer = item.querySelector('.faq-answer');
        const toggle = item.querySelector('.faq-toggle');
        
        question.addEventListener('click', () => {
            const isOpen = item.classList.contains('open');
            
            // Close all other FAQ items
            faqItems.forEach(otherItem => {
                if (otherItem !== item) {
                    otherItem.classList.remove('open');
                    const otherAnswer = otherItem.querySelector('.faq-answer');
                    otherAnswer.style.maxHeight = '0';
                }
            });
            
            // Toggle current item
            if (isOpen) {
                item.classList.remove('open');
                answer.style.maxHeight = '0';
            } else {
                item.classList.add('open');
                answer.style.maxHeight = answer.scrollHeight + 'px';
            }
        });
    });
}

// Form validation initialization
function initFormValidation() {
    // Add custom validation styles
    const style = document.createElement('style');
    style.textContent = `
        .form-group input.error,
        .form-group select.error,
        .form-group textarea.error {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(220, 53, 69, 0.1);
        }
        
        .form-error {
            display: none;
            color: var(--accent-color);
            font-size: var(--font-size-sm);
            margin-top: var(--spacing-xs);
        }
        
        .form-message {
            padding: var(--spacing-lg);
            border-radius: var(--radius-lg);
            margin-bottom: var(--spacing-xl);
            animation: slideInDown 0.3s ease-out;
        }
        
        .form-message.success {
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
        }
        
        .form-message.error {
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
        }
        
        .message-content {
            display: flex;
            align-items: flex-start;
            gap: var(--spacing-md);
        }
        
        .message-icon {
            flex-shrink: 0;
            margin-top: 2px;
        }
        
        .message-text {
            flex: 1;
        }
        
        @keyframes slideInDown {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .faq-item {
            border: 1px solid var(--border-color);
            border-radius: var(--radius-lg);
            margin-bottom: var(--spacing-md);
            overflow: hidden;
            transition: var(--transition-base);
        }
        
        .faq-item:hover {
            box-shadow: var(--shadow-md);
        }
        
        .faq-question {
            padding: var(--spacing-lg);
            cursor: pointer;
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--bg-primary);
            transition: var(--transition-base);
        }
        
        .faq-question:hover {
            background-color: var(--bg-secondary);
        }
        
        .faq-question h3 {
            margin: 0;
            font-size: var(--font-size-lg);
            color: var(--text-primary);
        }
        
        .faq-toggle {
            background: none;
            border: none;
            cursor: pointer;
            padding: var(--spacing-xs);
            border-radius: var(--radius-sm);
            transition: var(--transition-base);
            color: var(--text-secondary);
        }
        
        .faq-toggle:hover {
            background-color: var(--bg-secondary);
            color: var(--primary-color);
        }
        
        .faq-item.open .faq-toggle {
            transform: rotate(180deg);
        }
        
        .faq-answer {
            max-height: 0;
            overflow: hidden;
            transition: max-height 0.3s ease-out;
        }
        
        .faq-answer p {
            padding: 0 var(--spacing-lg) var(--spacing-lg);
            margin: 0;
            color: var(--text-secondary);
            line-height: 1.6;
        }
        
        .spinner {
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
    `;
    document.head.appendChild(style);
}

// Track form submission for analytics
function trackFormSubmission(data) {
    // Google Analytics or other tracking
    if (typeof gtag !== 'undefined') {
        gtag('event', 'form_submit', {
            event_category: 'Contact',
            event_label: data.subject,
            value: 1
        });
    }
    
    // Console log for development
    console.log('Form submitted:', {
        subject: data.subject,
        budget: data.budget,
        timestamp: new Date().toISOString()
    });
}

// Auto-save form data to localStorage
function initAutoSave() {
    const form = document.getElementById('contact-form');
    if (!form) return;
    
    const inputs = form.querySelectorAll('input, select, textarea');
    
    // Load saved data
    inputs.forEach(input => {
        const savedValue = localStorage.getItem(`contact_form_${input.name}`);
        if (savedValue && input.type !== 'checkbox') {
            input.value = savedValue;
        } else if (savedValue && input.type === 'checkbox') {
            input.checked = savedValue === 'true';
        }
    });
    
    // Save data on input
    inputs.forEach(input => {
        input.addEventListener('input', () => {
            if (input.type === 'checkbox') {
                localStorage.setItem(`contact_form_${input.name}`, input.checked);
            } else {
                localStorage.setItem(`contact_form_${input.name}`, input.value);
            }
        });
    });
    
    // Clear saved data on successful submission
    form.addEventListener('submit', () => {
        setTimeout(() => {
            inputs.forEach(input => {
                localStorage.removeItem(`contact_form_${input.name}`);
            });
        }, 3000); // Clear after success message
    });
}

// Initialize auto-save
document.addEventListener('DOMContentLoaded', initAutoSave);

// Export functions for global use
window.ContactForm = {
    showFormMessage,
    validateForm,
    isValidEmail,
    isValidPhone
};
