<?php
require_once __DIR__ . '/../config/database.php';

// Helper function for safe output with entity decoding
function safeOutputWithDecode($data) {
    if (empty($data)) return '';
    // First decode any existing HTML entities, then encode for safe output
    $decoded = html_entity_decode($data, ENT_QUOTES, 'UTF-8');
    return htmlspecialchars($decoded, ENT_QUOTES, 'UTF-8');
}

function getSiteSetting($key, $default = '') {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->prepare("SELECT setting_value FROM site_settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        $result = $stmt->fetch();
        return $result ? $result['setting_value'] : $default;
    } catch(PDOException $e) {
        error_log("Error fetching site setting: " . $e->getMessage());
        return $default;
    }
}

function getAboutData() {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->query("SELECT * FROM about_section ORDER BY id DESC LIMIT 1");
        return $stmt->fetch();
    } catch(PDOException $e) {
        error_log("Error fetching about data: " . $e->getMessage());
        return null;
    }
}

function getContactInfo() {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->query("SELECT * FROM contact_info ORDER BY id DESC LIMIT 1");
        return $stmt->fetch();
    } catch(PDOException $e) {
        error_log("Error fetching contact info: " . $e->getMessage());
        return null;
    }
}

function getSkills($category = null) {
    $pdo = getDBConnection();
    
    $sql = "SELECT * FROM skills";
    $params = [];
    
    if ($category) {
        $sql .= " WHERE category = ?";
        $params[] = $category;
    }
    
    $sql .= " ORDER BY category, display_order ASC";
    
    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    } catch(PDOException $e) {
        error_log("Error fetching skills: " . $e->getMessage());
        return [];
    }
}

function getSkillsByCategory() {
    $skills = getSkills();
    $grouped = [];
    
    foreach ($skills as $skill) {
        $grouped[$skill['category']][] = $skill;
    }
    
    return $grouped;
}

function getExperience() {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->query("SELECT * FROM experience ORDER BY display_order ASC, start_date DESC");
        return $stmt->fetchAll();
    } catch(PDOException $e) {
        error_log("Error fetching experience: " . $e->getMessage());
        return [];
    }
}

function addNewsletterSubscriber($email) {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->prepare("INSERT INTO newsletter_subscribers (email) VALUES (?) ON DUPLICATE KEY UPDATE status = 'active', subscribed_at = CURRENT_TIMESTAMP");
        return $stmt->execute([$email]);
    } catch(PDOException $e) {
        error_log("Error adding newsletter subscriber: " . $e->getMessage());
        return false;
    }
}

function saveContactMessage($data) {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->prepare("INSERT INTO contact_messages (name, email, phone, subject, budget, message, newsletter_signup) VALUES (?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([
            $data['name'],
            $data['email'],
            $data['phone'] ?? null,
            $data['subject'],
            $data['budget'] ?? null,
            $data['message'],
            $data['newsletter_signup'] ?? false
        ]);
    } catch(PDOException $e) {
        error_log("Error saving contact message: " . $e->getMessage());
        return false;
    }
}

function formatDate($date, $format = 'M Y') {
    if (!$date) return '';
    return date($format, strtotime($date));
}

function formatExperienceDate($start_date, $end_date, $is_current = false) {
    $start = formatDate($start_date);
    
    if ($is_current) {
        return $start . ' - Present';
    } elseif ($end_date) {
        return $start . ' - ' . formatDate($end_date);
    } else {
        return $start;
    }
}

// Admin helper functions
function updateSiteSetting($key, $value, $type = 'text') {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->prepare("INSERT INTO site_settings (setting_key, setting_value, setting_type) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
        return $stmt->execute([$key, $value, $type, $value]);
    } catch(PDOException $e) {
        error_log("Error updating site setting: " . $e->getMessage());
        return false;
    }
}

function updateAboutSection($data) {
    $pdo = getDBConnection();
    
    try {
        // Check if about section exists
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM about_section");
        $count = $stmt->fetch()['count'];
        
        if ($count > 0) {
            // Update existing record
            $sql = "UPDATE about_section SET name = ?, title = ?, bio = ?, hero_description = ?, story_content = ?, years_experience = ?, projects_completed = ?, happy_clients = ?, profile_image = ? WHERE id = (SELECT id FROM (SELECT id FROM about_section ORDER BY id DESC LIMIT 1) as temp)";
        } else {
            // Insert new record
            $sql = "INSERT INTO about_section (name, title, bio, hero_description, story_content, years_experience, projects_completed, happy_clients, profile_image) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
        }
        
        $stmt = $pdo->prepare($sql);
        return $stmt->execute([
            $data['name'],
            $data['title'],
            $data['bio'],
            $data['hero_description'],
            $data['story_content'],
            $data['years_experience'],
            $data['projects_completed'],
            $data['happy_clients'],
            $data['profile_image']
        ]);
    } catch(PDOException $e) {
        error_log("Error updating about section: " . $e->getMessage());
        return false;
    }
}

function getContactMessages($limit = null, $offset = 0, $status = null) {
    $pdo = getDBConnection();
    
    try {
        $where_clause = '';
        $params = [];
        
        if ($status) {
            $where_clause = 'WHERE status = ?';
            $params[] = $status;
        }
        
        $limit_clause = '';
        if ($limit) {
            $limit_clause = "LIMIT $offset, $limit";
        }
        
        $stmt = $pdo->prepare("SELECT * FROM contact_messages $where_clause ORDER BY created_at DESC $limit_clause");
        $stmt->execute($params);
        return $stmt->fetchAll();
    } catch(PDOException $e) {
        error_log("Error fetching contact messages: " . $e->getMessage());
        return [];
    }
}

function getNewsletterSubscribers($limit = null, $offset = 0, $status = null) {
    $pdo = getDBConnection();
    
    try {
        $where_clause = '';
        $params = [];
        
        if ($status) {
            $where_clause = 'WHERE status = ?';
            $params[] = $status;
        }
        
        $limit_clause = '';
        if ($limit) {
            $limit_clause = "LIMIT $offset, $limit";
        }
        
        $stmt = $pdo->prepare("SELECT * FROM newsletter_subscribers $where_clause ORDER BY subscribed_at DESC $limit_clause");
        $stmt->execute($params);
        return $stmt->fetchAll();
    } catch(PDOException $e) {
        error_log("Error fetching newsletter subscribers: " . $e->getMessage());
        return [];
    }
}

function deleteContactMessage($message_id) {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->prepare("DELETE FROM contact_messages WHERE id = ?");
        return $stmt->execute([$message_id]);
    } catch(PDOException $e) {
        error_log("Error deleting contact message: " . $e->getMessage());
        return false;
    }
}

function updateContactMessageStatus($message_id, $status) {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->prepare("UPDATE contact_messages SET status = ? WHERE id = ?");
        return $stmt->execute([$status, $message_id]);
    } catch(PDOException $e) {
        error_log("Error updating contact message status: " . $e->getMessage());
        return false;
    }
}

function updateNewsletterSubscriberStatus($subscriber_id, $status) {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->prepare("UPDATE newsletter_subscribers SET status = ? WHERE id = ?");
        return $stmt->execute([$status, $subscriber_id]);
    } catch(PDOException $e) {
        error_log("Error updating newsletter subscriber status: " . $e->getMessage());
        return false;
    }
}

function deleteNewsletterSubscriber($subscriber_id) {
    $pdo = getDBConnection();
    
    try {
        $stmt = $pdo->prepare("DELETE FROM newsletter_subscribers WHERE id = ?");
        return $stmt->execute([$subscriber_id]);
    } catch(PDOException $e) {
        error_log("Error deleting newsletter subscriber: " . $e->getMessage());
        return false;
    }
}
?>
