<?php
require_once '../config/auth.php';
require_once '../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$pdo = getDBConnection();
$message = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add' || $action === 'edit') {
        $id = $_POST['id'] ?? null;
        $title = sanitizeInput($_POST['title'] ?? '');
        $description = sanitizeInput($_POST['description'] ?? '');
        $category = sanitizeInput($_POST['category'] ?? '');
        $client = sanitizeInput($_POST['client'] ?? '');
        $year = (int)($_POST['year'] ?? date('Y'));
        $tools_used = sanitizeInput($_POST['tools_used'] ?? '');
        $is_featured = isset($_POST['is_featured']) ? 1 : 0;
        $status = $_POST['status'] ?? 'active';
        
        // Handle image upload
        $image_path = '';
        if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
            $upload_dir = '../uploads/portfolio/';
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            $file_extension = strtolower(pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION));
            $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
            
            if (in_array($file_extension, $allowed_extensions)) {
                $filename = uniqid() . '.' . $file_extension;
                $upload_path = $upload_dir . $filename;
                
                if (move_uploaded_file($_FILES['image']['tmp_name'], $upload_path)) {
                    $image_path = 'uploads/portfolio/' . $filename;
                } else {
                    $error = 'Failed to upload image.';
                }
            } else {
                $error = 'Invalid file type. Please upload JPG, PNG, GIF, or WebP images.';
            }
        }
        
        if (empty($error)) {
            try {
                if ($action === 'add') {
                    $sql = "INSERT INTO portfolio_items (title, description, category, client, year, tools_used, image_path, is_featured, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute([$title, $description, $category, $client, $year, $tools_used, $image_path, $is_featured, $status]);
                    $message = 'Portfolio item added successfully!';
                } else {
                    $sql = "UPDATE portfolio_items SET title = ?, description = ?, category = ?, client = ?, year = ?, tools_used = ?, is_featured = ?, status = ?";
                    $params = [$title, $description, $category, $client, $year, $tools_used, $is_featured, $status];
                    
                    if ($image_path) {
                        $sql .= ", image_path = ?";
                        $params[] = $image_path;
                    }
                    
                    $sql .= " WHERE id = ?";
                    $params[] = $id;
                    
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute($params);
                    $message = 'Portfolio item updated successfully!';
                }
            } catch(PDOException $e) {
                $error = 'Database error: ' . $e->getMessage();
            }
        }
    } elseif ($action === 'delete') {
        $id = (int)($_POST['id'] ?? 0);
        try {
            // Get image path to delete file
            $stmt = $pdo->prepare("SELECT image_path FROM portfolio_items WHERE id = ?");
            $stmt->execute([$id]);
            $item = $stmt->fetch();
            
            if ($item && $item['image_path'] && file_exists('../' . $item['image_path'])) {
                unlink('../' . $item['image_path']);
            }
            
            $stmt = $pdo->prepare("DELETE FROM portfolio_items WHERE id = ?");
            $stmt->execute([$id]);
            $message = 'Portfolio item deleted successfully!';
        } catch(PDOException $e) {
            $error = 'Error deleting item: ' . $e->getMessage();
        }
    }
}

// Get portfolio items
try {
    $stmt = $pdo->query("SELECT * FROM portfolio_items ORDER BY display_order ASC, created_at DESC");
    $portfolio_items = $stmt->fetchAll();
} catch(PDOException $e) {
    $error = 'Error fetching portfolio items: ' . $e->getMessage();
    $portfolio_items = [];
}

// Get item for editing
$edit_item = null;
if (isset($_GET['edit'])) {
    $edit_id = (int)$_GET['edit'];
    try {
        $stmt = $pdo->prepare("SELECT * FROM portfolio_items WHERE id = ?");
        $stmt->execute([$edit_id]);
        $edit_item = $stmt->fetch();
    } catch(PDOException $e) {
        $error = 'Error fetching item for editing: ' . $e->getMessage();
    }
}

$user = $auth->getCurrentUser();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Portfolio Management - Admin</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="assets/admin.css">
</head>
<body>
    <div class="admin-wrapper">
        <!-- Sidebar -->
        <nav class="sidebar">
            <div class="sidebar-header">
                <h2>Portfolio Admin</h2>
            </div>
            <ul class="sidebar-menu">
                <li><a href="index.php"><i>📊</i> Dashboard</a></li>
                <li><a href="about.php"><i>👤</i> About Section</a></li>
                <li><a href="portfolio.php" class="active"><i>🎨</i> Portfolio</a></li>
                <li><a href="contact.php"><i>📞</i> Contact Info</a></li>
                <li><a href="messages.php"><i>💬</i> Messages</a></li>
                <li><a href="newsletter.php"><i>📧</i> Newsletter</a></li>
                <li><a href="settings.php"><i>⚙️</i> Settings</a></li>
                <li><a href="logout.php"><i>🚪</i> Logout</a></li>
            </ul>
        </nav>

        <!-- Main Content -->
        <div class="main-content">
            <header class="content-header">
                <h1>Portfolio Management</h1>
                <div class="user-info">
                    Welcome, <?php echo htmlspecialchars($user['username']); ?>
                </div>
            </header>

            <div class="dashboard-content">
                <?php if ($message): ?>
                    <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
                <?php endif; ?>

                <!-- Add/Edit Form -->
                <div class="form-container">
                    <h3><?php echo $edit_item ? 'Edit Portfolio Item' : 'Add New Portfolio Item'; ?></h3>
                    <form method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="<?php echo $edit_item ? 'edit' : 'add'; ?>">
                        <?php if ($edit_item): ?>
                            <input type="hidden" name="id" value="<?php echo $edit_item['id']; ?>">
                        <?php endif; ?>
                        
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="title">Project Title *</label>
                                <input type="text" id="title" name="title" value="<?php echo htmlspecialchars($edit_item['title'] ?? ''); ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="category">Category *</label>
                                <select id="category" name="category" required>
                                    <option value="">Select Category</option>
                                    <option value="pamphlets" <?php echo ($edit_item['category'] ?? '') === 'pamphlets' ? 'selected' : ''; ?>>Pamphlet Designs</option>
                                    <option value="events" <?php echo ($edit_item['category'] ?? '') === 'events' ? 'selected' : ''; ?>>Event Welcome Signs</option>
                                    <option value="promotions" <?php echo ($edit_item['category'] ?? '') === 'promotions' ? 'selected' : ''; ?>>Restaurant & School</option>
                                    <option value="brands" <?php echo ($edit_item['category'] ?? '') === 'brands' ? 'selected' : ''; ?>>Brand Posters</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="client">Client</label>
                                <input type="text" id="client" name="client" value="<?php echo htmlspecialchars($edit_item['client'] ?? ''); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label for="year">Year</label>
                                <input type="number" id="year" name="year" value="<?php echo $edit_item['year'] ?? date('Y'); ?>" min="2000" max="<?php echo date('Y') + 1; ?>">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Description</label>
                            <textarea id="description" name="description" rows="4"><?php echo htmlspecialchars($edit_item['description'] ?? ''); ?></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="tools_used">Tools Used</label>
                            <input type="text" id="tools_used" name="tools_used" value="<?php echo htmlspecialchars($edit_item['tools_used'] ?? 'Adobe Photoshop, Illustrator'); ?>" placeholder="e.g., Adobe Photoshop, Illustrator">
                        </div>
                        
                        <div class="form-group">
                            <label for="image">Project Image</label>
                            <input type="file" id="image" name="image" accept="image/*">
                            <?php if ($edit_item && $edit_item['image_path']): ?>
                                <div class="current-image">
                                    <p>Current image:</p>
                                    <img src="../<?php echo htmlspecialchars($edit_item['image_path']); ?>" alt="Current image" class="image-preview">
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="form-grid">
                            <div class="form-group">
                                <label>
                                    <input type="checkbox" name="is_featured" <?php echo ($edit_item['is_featured'] ?? 0) ? 'checked' : ''; ?>>
                                    Featured Item
                                </label>
                            </div>
                            
                            <div class="form-group">
                                <label for="status">Status</label>
                                <select id="status" name="status">
                                    <option value="active" <?php echo ($edit_item['status'] ?? 'active') === 'active' ? 'selected' : ''; ?>>Active</option>
                                    <option value="inactive" <?php echo ($edit_item['status'] ?? '') === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary">
                                <?php echo $edit_item ? 'Update Item' : 'Add Item'; ?>
                            </button>
                            <?php if ($edit_item): ?>
                                <a href="portfolio.php" class="btn btn-secondary">Cancel</a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>

                <!-- Portfolio Items List -->
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Image</th>
                                <th>Title</th>
                                <th>Category</th>
                                <th>Client</th>
                                <th>Year</th>
                                <th>Featured</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($portfolio_items)): ?>
                                <tr>
                                    <td colspan="8" class="no-data">No portfolio items found</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach($portfolio_items as $item): ?>
                                    <tr>
                                        <td>
                                            <?php if ($item['image_path']): ?>
                                                <img src="../<?php echo htmlspecialchars($item['image_path']); ?>" alt="<?php echo htmlspecialchars($item['title']); ?>" style="width: 60px; height: 60px; object-fit: cover; border-radius: 5px;">
                                            <?php else: ?>
                                                <div style="width: 60px; height: 60px; background: #f0f0f0; border-radius: 5px; display: flex; align-items: center; justify-content: center; font-size: 0.8rem; color: #999;">No Image</div>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($item['title']); ?></td>
                                        <td><?php echo ucfirst(htmlspecialchars($item['category'])); ?></td>
                                        <td><?php echo htmlspecialchars($item['client'] ?: 'N/A'); ?></td>
                                        <td><?php echo $item['year']; ?></td>
                                        <td>
                                            <span class="status-badge <?php echo $item['is_featured'] ? 'status-active' : 'status-inactive'; ?>">
                                                <?php echo $item['is_featured'] ? 'Yes' : 'No'; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="status-badge status-<?php echo $item['status']; ?>">
                                                <?php echo ucfirst($item['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <div class="table-actions">
                                                <a href="portfolio.php?edit=<?php echo $item['id']; ?>" class="btn btn-secondary">Edit</a>
                                                <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this item?');">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="id" value="<?php echo $item['id']; ?>">
                                                    <button type="submit" class="btn btn-danger">Delete</button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
