<?php
require_once __DIR__ . '/../config/auth.php';
require_once __DIR__ . '/../config/database.php';

$auth = new Auth();
$auth->requireLogin();

$pdo = getDBConnection();
$success = '';
$error = '';

// Handle message actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $message_id = (int)($_POST['message_id'] ?? 0);
    
    if ($action === 'delete' && $message_id > 0) {
        try {
            $stmt = $pdo->prepare("DELETE FROM contact_messages WHERE id = ?");
            if ($stmt->execute([$message_id])) {
                $success = 'Message deleted successfully!';
            } else {
                $error = 'Error deleting message.';
            }
        } catch(PDOException $e) {
            $error = 'Database error: ' . $e->getMessage();
        }
    } elseif ($action === 'mark_read' && $message_id > 0) {
        try {
            $stmt = $pdo->prepare("UPDATE contact_messages SET status = 'read' WHERE id = ?");
            if ($stmt->execute([$message_id])) {
                $success = 'Message marked as read!';
            } else {
                $error = 'Error updating message status.';
            }
        } catch(PDOException $e) {
            $error = 'Database error: ' . $e->getMessage();
        }
    } elseif ($action === 'mark_replied' && $message_id > 0) {
        try {
            $stmt = $pdo->prepare("UPDATE contact_messages SET status = 'replied' WHERE id = ?");
            if ($stmt->execute([$message_id])) {
                $success = 'Message marked as replied!';
            } else {
                $error = 'Error updating message status.';
            }
        } catch(PDOException $e) {
            $error = 'Database error: ' . $e->getMessage();
        }
    }
}

// Get messages with pagination
$page = (int)($_GET['page'] ?? 1);
$per_page = 20;
$offset = ($page - 1) * $per_page;

try {
    // Get total count
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM contact_messages");
    $total_messages = $stmt->fetch()['count'];
    $total_pages = ceil($total_messages / $per_page);
    
    // Get messages for current page
    $stmt = $pdo->prepare("SELECT * FROM contact_messages ORDER BY created_at DESC LIMIT $offset, $per_page");
    $stmt->execute();
    $messages = $stmt->fetchAll();
    
    // Get statistics
    $stmt = $pdo->query("SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'new' THEN 1 ELSE 0 END) as new_count,
        SUM(CASE WHEN status = 'read' THEN 1 ELSE 0 END) as read_count,
        SUM(CASE WHEN status = 'replied' THEN 1 ELSE 0 END) as replied_count
        FROM contact_messages");
    $stats = $stmt->fetch();
    
} catch(PDOException $e) {
    $error = 'Error fetching messages: ' . $e->getMessage();
    $messages = [];
    $stats = ['total' => 0, 'new_count' => 0, 'read_count' => 0, 'replied_count' => 0];
}

$user = $auth->getCurrentUser();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Messages - Admin Panel</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="assets/admin.css">
</head>
<body>
    <div class="admin-wrapper">
        <!-- Sidebar -->
        <nav class="sidebar">
            <div class="sidebar-header">
                <h2>Portfolio Admin</h2>
            </div>
            <ul class="sidebar-menu">
                <li><a href="index.php"><i>📊</i> Dashboard</a></li>
                <li><a href="about.php"><i>👤</i> About Section</a></li>
                <li><a href="portfolio.php"><i>🎨</i> Portfolio</a></li>
                <li><a href="contact.php"><i>📞</i> Contact Info</a></li>
                <li><a href="messages.php" class="active"><i>💬</i> Messages</a></li>
                <li><a href="newsletter.php"><i>📧</i> Newsletter</a></li>
                <li><a href="settings.php"><i>⚙️</i> Settings</a></li>
                <li><a href="logout.php"><i>🚪</i> Logout</a></li>
            </ul>
        </nav>

        <!-- Main Content -->
        <div class="main-content">
            <header class="content-header">
                <h1>Contact Messages</h1>
                <div class="user-info">
                    Welcome, <?php echo htmlspecialchars($user['username']); ?>
                </div>
            </header>

            <div class="dashboard-content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
                <?php endif; ?>

                <!-- Statistics -->
                <div class="stats-grid">
                    <div class="stat-card">
                        <div class="stat-icon">💬</div>
                        <div class="stat-info">
                            <h3><?php echo $stats['total']; ?></h3>
                            <p>Total Messages</p>
                        </div>
                    </div>
                    
                    <div class="stat-card">
                        <div class="stat-icon">🆕</div>
                        <div class="stat-info">
                            <h3><?php echo $stats['new_count']; ?></h3>
                            <p>New Messages</p>
                        </div>
                    </div>
                    
                    <div class="stat-card">
                        <div class="stat-icon">👁️</div>
                        <div class="stat-info">
                            <h3><?php echo $stats['read_count']; ?></h3>
                            <p>Read Messages</p>
                        </div>
                    </div>
                    
                    <div class="stat-card">
                        <div class="stat-icon">✅</div>
                        <div class="stat-info">
                            <h3><?php echo $stats['replied_count']; ?></h3>
                            <p>Replied Messages</p>
                        </div>
                    </div>
                </div>

                <!-- Messages Table -->
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th>Email</th>
                                <th>Subject</th>
                                <th>Date</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($messages)): ?>
                                <tr>
                                    <td colspan="6" class="no-data">No messages found</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach($messages as $message): ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo htmlspecialchars($message['name']); ?></strong>
                                            <?php if ($message['phone']): ?>
                                                <br><small style="color: #666;"><?php echo htmlspecialchars($message['phone']); ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($message['email']); ?></td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($message['subject']); ?></strong>
                                            <?php if ($message['budget']): ?>
                                                <br><small style="color: #666;">Budget: <?php echo htmlspecialchars($message['budget']); ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo date('M j, Y g:i A', strtotime($message['created_at'])); ?></td>
                                        <td>
                                            <span class="status-badge status-<?php echo $message['status']; ?>">
                                                <?php echo ucfirst($message['status']); ?>
                                            </span>
                                            <?php if ($message['newsletter_signup']): ?>
                                                <br><small style="color: #8B5A96;">📧 Newsletter signup</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="table-actions">
                                                <button onclick="showMessage(<?php echo $message['id']; ?>)" class="btn btn-primary" style="padding: 0.5rem 1rem; font-size: 0.9rem;">View</button>
                                                
                                                <?php if ($message['status'] === 'new'): ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="action" value="mark_read">
                                                        <input type="hidden" name="message_id" value="<?php echo $message['id']; ?>">
                                                        <button type="submit" class="btn btn-secondary" style="padding: 0.5rem 1rem; font-size: 0.9rem;">Mark Read</button>
                                                    </form>
                                                <?php endif; ?>
                                                
                                                <?php if ($message['status'] !== 'replied'): ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="action" value="mark_replied">
                                                        <input type="hidden" name="message_id" value="<?php echo $message['id']; ?>">
                                                        <button type="submit" class="btn btn-secondary" style="padding: 0.5rem 1rem; font-size: 0.9rem;">Mark Replied</button>
                                                    </form>
                                                <?php endif; ?>
                                                
                                                <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this message?')">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="message_id" value="<?php echo $message['id']; ?>">
                                                    <button type="submit" class="btn btn-danger" style="padding: 0.5rem 1rem; font-size: 0.9rem;">Delete</button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                <div style="margin-top: 2rem; text-align: center;">
                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                        <?php if ($i === $page): ?>
                            <span style="padding: 0.5rem 1rem; background: #8B5A96; color: white; border-radius: 4px; margin: 0 0.25rem;"><?php echo $i; ?></span>
                        <?php else: ?>
                            <a href="?page=<?php echo $i; ?>" style="padding: 0.5rem 1rem; background: #f8f9fa; color: #333; text-decoration: none; border-radius: 4px; margin: 0 0.25rem;"><?php echo $i; ?></a>
                        <?php endif; ?>
                    <?php endfor; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Message Modal -->
    <div id="messageModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 2000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: white; padding: 2rem; border-radius: 10px; max-width: 600px; width: 90%; max-height: 80vh; overflow-y: auto;">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem;">
                <h3>Message Details</h3>
                <button onclick="closeModal()" style="background: none; border: none; font-size: 1.5rem; cursor: pointer;">&times;</button>
            </div>
            <div id="messageContent"></div>
        </div>
    </div>

    <script>
        function showMessage(messageId) {
            // Find message data
            const messages = <?php echo json_encode($messages); ?>;
            const message = messages.find(m => m.id == messageId);
            
            if (message) {
                const content = `
                    <div style="margin-bottom: 1rem;">
                        <strong>From:</strong> ${message.name}<br>
                        <strong>Email:</strong> ${message.email}<br>
                        ${message.phone ? `<strong>Phone:</strong> ${message.phone}<br>` : ''}
                        <strong>Subject:</strong> ${message.subject}<br>
                        ${message.budget ? `<strong>Budget:</strong> ${message.budget}<br>` : ''}
                        <strong>Date:</strong> ${new Date(message.created_at).toLocaleString()}<br>
                        <strong>Status:</strong> ${message.status}<br>
                        ${message.newsletter_signup ? '<strong>Newsletter:</strong> Yes<br>' : ''}
                    </div>
                    <div style="margin-top: 1rem;">
                        <strong>Message:</strong>
                        <div style="background: #f8f9fa; padding: 1rem; border-radius: 8px; margin-top: 0.5rem; white-space: pre-wrap;">${message.message}</div>
                    </div>
                `;
                document.getElementById('messageContent').innerHTML = content;
                document.getElementById('messageModal').style.display = 'block';
            }
        }
        
        function closeModal() {
            document.getElementById('messageModal').style.display = 'none';
        }
        
        // Close modal when clicking outside
        document.getElementById('messageModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });
    </script>
</body>
</html>
